% main script for load flow simulation

clear all;
close all;

disp(' ');
disp('********************************************************');
disp('* This script solves the load flow problem *');
disp('********************************************************');
disp(' ');

%%  Step 1: load Y matrix & base values
%   The nodal admittance matrix should saved in a folder 'Data_LF'
%   located in the same directory as this script.

disp('STEP 1: loading nodal admittance matrix and base values ...');
disp(' ');

% ! Define the network to use !
network = 'small'; % either 'small' or 'big'

% choose correct .mat files
switch(network)
    case 'small'
        Y_file  = './Data_LF/Small Network/Y.mat';
        Ab_file = './Data_LF/Small Network/Base_Power.mat';
        Vb_file = './Data_LF/Small Network/Base_Voltage.mat';
    case 'big'
        Y_file  = './Data_LF/Big Network/Y.mat';
        Ab_file = './Data_LF/Big Network/Base_Power.mat';
        Vb_file = './Data_LF/Big Network/Base_Voltage.mat';        
end


% Put the name of the file containing the Y matrix
Y = importdata(Y_file);

% Put the names of the files containing the base values
Ab = importdata(Ab_file);
Vb = importdata(Vb_file);

n_nodes = size(Y,1); % number of nodes

disp(['The network consists of ',num2str(n_nodes), ' nodes.']);
disp(['The base power is ' num2str(Ab/1e6), 'MVA.']);
disp(['The base voltage is ' num2str(Vb/1e3) 'kV.']);
disp(' ');

%%  Step 2: load power profiles of loads and generators
% The nodal power profiles should be also be saved 'Data_LF',
% as matrices of size n_nodes x n_timesteps.

disp('STEP 2: Loading power profiles.')
disp(' ');

% ! Define the profile type !
% if network = 'small' -> 'daily' or 'single',
% if network = 'big' -> 'daily',
profile_type = 'daily'; % either 'single' or 'daily'

% load
switch(profile_type)
    case 'single'
        P_load = importdata('./Data_LF/Small Network/P_load_single_value.mat');
        Q_load = importdata('./Data_LF/Small Network/Q_load_single_value.mat');

        P_gen = importdata('./Data_LF/Small Network/P_gen_single_value.mat');
        Q_gen = importdata('./Data_LF/Small Network/Q_gen_single_value.mat');
    case 'daily'
        switch(network)
            case 'big'
                P_load = importdata('./Data_LF/Big Network/P_daily_load_curve.mat');
                Q_load = importdata('./Data_LF/Big Network/Q_daily_load_curve.mat');

                P_gen = importdata('./Data_LF/Big Network/P_daily_gen_curve.mat');
                Q_gen = importdata('./Data_LF/Big Network/Q_daily_gen_curve.mat');
            case'small'
                P_load = importdata('./Data_LF/Small Network/P_daily_load_curve.mat');
                Q_load = importdata('./Data_LF/Small Network/Q_daily_load_curve.mat');

                P_gen = importdata('./Data_LF/Small Network/P_daily_gen_curve.mat');
                Q_gen = importdata('./Data_LF/Small Network/Q_daily_gen_curve.mat');
            otherwise
                error('unknown network type');
        end
    otherwise
        error('unknown profile type');
end

% sanity check
if (~all([size(P_load,1),size(Q_load,1),size(P_gen,1),size(Q_gen,1)]==n_nodes))
    error('The P/Q profiles are not compatible in terms of number of nodes.');
elseif (length(unique([size(P_load,2),size(Q_load,2),size(P_gen,2),size(Q_gen,2)]))~=1)
    error('The P/Q profiles are not compatible in terms of number of timesteps.');
end

n_timesteps = size(P_load,2);

disp(['The number of timesteps is ' num2str(n_timesteps) '.']);
disp(' ');

% Scale factor for load and generations of Active powers
gen_scale = 1; load_scale = 1;
P_gen = gen_scale*P_gen;
P_load = load_scale*P_load;

% Complex power in p.u. (net)
S_star = complex(P_gen-P_load, Q_gen-Q_load) / Ab;
% S_star = complex(P_inject-P_absorb,P_absorb - P_inject) / A_b;
% S_star = complex(P_inject-P_absorb,0 - 0) / A_b;
EMTP_Loads.P = real(S_star*Ab/3);
EMTP_Loads.Q = imag(S_star*Ab/3);

%%  Step 3: Simulation parameters
%%% Set the complex load values in p.u. and initialize the bus voltages
%%% Also define tolerance and max number of allowed iterations for the NR

disp('STEP 3: Configuring the load flow simulation.');
disp(' ');

% Configure the bus types
switch(network)
    case 'small'
        % index of the slack bus
        idx.slack = 1;
        % indices of the PV buses
        idx.pv = [];
        % indices of the PQ buses
        idx.pq = (2:n_nodes)';
    case 'big'
        % index of the slack bus
        idx.slack = 1;
        % indices of the PV buses
        idx.pv = [4, 9, 17, 25, 33].';
        % indices of the PQ buses
        idx.loads = [5, 12, 14, 18, 22, 24, 29, 32, 34].'; % nodes with loads
        idx.gen   = [7, 15, 20, 27, 30].';                 % nodes with generation
        idx.zero  = (1:size(Y,1)).';                       % PQ nodes with no injections (zero-injection nodes)
        idx.zero([idx.slack; idx.pv; idx.loads; idx.gen]) = [];
        idx.pq = sort([idx.loads; idx.gen; idx.zero]);
end

% ! Customize the load flow simulation !

% Choose LF formulation either 'rectangular' or 'polar'.
coordinate_type = 'polar';

% if coordinate_type='daily' -> 'flat' or 'previous',
% if coordinate_type='single' -> 'flat' or 'bad'
start_type = 'previous';

% ! Enter Newton-Raphson algorithm parameters !

% maximum number of iterations
Parameters.n_max = 100;
% convergence tolerance
Parameters.tol = 1e-7;

%% Step 4: Newton-Raphson algorithm

disp('STEP 4: running Newton-Raphson algorithm.');
disp(' ');

% initialize
E_star = zeros(size(S_star)); E_star(idx.pv,:) = 1; % we assume here that all PV nodes have 1 p.u. as voltage reference
E = zeros(n_nodes,n_timesteps);
S = zeros(n_nodes,n_timesteps);
n_iter = zeros(1,n_timesteps);
t_exec = zeros(1,n_timesteps);

for k = 1:n_timesteps
    %% Initialization
    
    switch(start_type)
        case 'flat'
            % always use a flat start
            E_0 = ones(n_nodes,1);
        case 'previous'
            % use the result of a previous timestep (if available)
            if(k<=1)
                E_0 = ones(n_nodes,1);
            else
                E_0 = E(:,k-1);
            end
        case 'bad'
            % bad start (only for the case 'single')
            E_0 = ones(n_nodes,1);
            E_0(3) = exp(-1i*pi/4);
        otherwise
            error('unknown start type');
    end
    
    %% Call Newton-Raphson function
    
    % The functions NR_rectangular/NR_polar(.) internally make use of
    % rectangular/polar coordinates for the calculation.
    % However, they accept and return complex values (i.e., phasors).
    
    t_start = tic;
    
    switch(coordinate_type)
        case 'rectangular'
            [J,E(:,k),S(:,k),n_iter(k)] = NR_rectangular(S_star(:,k),E_star(:,k),Y,E_0,idx,Parameters);
        case 'polar'
            [J,E(:,k),S(:,k),n_iter(k)] = NR_polar(S_star(:,k),E_star(:,k),Y,E_0,idx,Parameters);
    end
    
    t_exec(k) = toc(t_start);
end

%% Step 5: Visualization

switch(profile_type)
    case 'single'
        
        % Power profiles at a specific timestep
        figure(1);
        clf;

        axes('FontSize',18);
        hold on;
        plot((1:n_nodes),P_load(:,1),'--ro','LineWidth',2,'MarkerSize',6);
        plot((1:n_nodes),Q_load(:,1),'--bo','LineWidth',2,'MarkerSize',6);
        plot((1:n_nodes),P_gen(:,1),'--go','LineWidth',2,'MarkerSize',6);
        plot((1:n_nodes),Q_gen(:,1),'ko','LineWidth',2,'MarkerSize',6);
        hold off;

        legend('P_{load}','Q_{load}','P_{generation}','Q_{generation}','Location','SouthEast');
        title('Nodal Power Profiles at a Specific Timestep');
        xlabel('Bus Index');
        ylabel('Power');

        set(gca,'XTick',1:n_nodes);
        axis([1,n_nodes,-max(max(abs(P_load(:,1))),max(abs(P_gen(:,1)))),Inf]);

        % Voltage magnitude profile at a specific timestep
        
        figure(2);
        clf;
        
        axes('FontSize',18);
        plot((1:n_nodes),abs(E),'--bs',...
            'LineWidth',2,...
            'MarkerSize',6,...
            'MarkerEdgeColor','b',...
            'MarkerFaceColor',[0.5,0.5,0.5]);
        title('Voltage Magnitude Profile at a Specific Timestep');
        xlabel('Bus Index');
        ylabel('Voltage Magnitude (p.u.)');
        set(gca,'XTick',1:n_nodes);
        
        % Phase angle profile at a specific timestep
        
        figure(3);
        clf;
        
        axes('FontSize',18);
        plot((1:n_nodes),rad2deg(angle(E)),'--cs',...
            'LineWidth',2,...
            'MarkerSize',6,...
            'MarkerEdgeColor','c',...
            'MarkerFaceColor',[0.5,0.5,0.5]);
        title('Phase Angle Profile at a Specific Timestep');
        xlabel('Bus Index');
        ylabel('Phase Angle (deg)');
        set(gca,'XTick',1:n_nodes);
        
    case 'daily'
        
        % Profiles over a day
        Time = ((1:size(S,2)).' - 1)*24/96;
        figure(2);
        clf;
        
        switch(network)
            case 'small'
                subplot(2,2,1);
                plot(Time, real(S)');
                title('Active Power (p.u.)'); xlabel('Time [h]');
                grid on; axis tight;

                subplot(2,2,2);
                plot(Time, imag(S)');
                title('Reactive Power (p.u.)'); xlabel('Time [h]');
                grid on; axis tight;

                subplot(2,2,3);
                plot(Time, abs(E)');
                title('Voltage Magnitude (p.u.)'); xlabel('Time [h]');
                grid on; axis tight;

                subplot(2,2,4);
                plot(Time, rad2deg(angle(E))');
                title('Phase Angle (deg)'); xlabel('Time [h]');
                grid on; axis tight;

                legend({'Bus 1','Bus 2','Bus 3','Bus 4','Bus 5','Bus 6'},'Location','NorthWest');
            case 'big'
                
                % Slack node
                subplot(3,4,1);
                plot(Time, real(S(idx.slack,:))');
                title('Slack Node - Active Power (p.u.)'); xlabel('Time [h]');
                grid on; axis tight;
                subplot(3,4,2);
                plot(Time, imag(S(idx.slack,:))');
                title('Slack Node - Reactive Power (p.u.)'); xlabel('Time [h]');
                grid on; axis tight; 
                subplot(3,4,3);
                plot(Time, abs(E(idx.slack,:))');
                title('Slack Node - Voltage Magnitude (p.u.)'); xlabel('Time [h]');
                grid on; axis tight;
                subplot(3,4,4);
                plot(Time, rad2deg(angle(E(idx.slack,:)))'); 
                title('Slack Node - Phase Angle (deg)'); xlabel('Time [h]');
                grid on; axis tight;  
                
                % PV nodes
                subplot(3,4,5);
                plot(Time, real(S(idx.pv,:))');
                title('PV Nodes - Active Power (p.u.)'); xlabel('Time [h]');
                grid on; axis tight;
                subplot(3,4,6);
                plot(Time, imag(S(idx.pv,:))');
                title('PV Nodes - Reactive Power (p.u.)'); xlabel('Time [h]');
                grid on; axis tight; 
                subplot(3,4,7);
                plot(Time, abs(E(idx.pv,:))');
                title('PV Nodes - Voltage Magnitude (p.u.)'); xlabel('Time [h]');
                grid on; axis tight; 
                subplot(3,4,8);
                plot(Time, rad2deg(angle(E(idx.pv,:)))'); 
                title('PV Nodes - Phase Angle (deg)'); xlabel('Time [h]');
                grid on; axis tight;
                
                % PQ nodes
                subplot(3,4,9);
                plot(Time, real(S(idx.pq,:))');
                title('PQ Nodes - Active Power (p.u.)'); xlabel('Time [h]');
                grid on; axis tight;
                subplot(3,4,10);
                plot(Time, imag(S(idx.pq,:))');
                title('PQ Nodes - Reactive Power (p.u.)'); xlabel('Time [h]');
                grid on; axis tight; 
                subplot(3,4,11);
                plot(Time, abs(E(idx.pq,:))');
                title('PQ Nodes  - Voltage Magnitude (p.u.)'); xlabel('Time [h]');
                grid on; axis tight;
                subplot(3,4,12);
                plot(Time, rad2deg(angle(E(idx.pq,:)))'); 
                title('PQ Nodes - Phase Angle (deg)'); xlabel('Time [h]');
                grid on; axis tight;                  
        end
        
        % Performance
        
        figure(3);
        clf;
        
        subplot(1,2,1);
        plot(n_iter');
        title('Number of Iterations');
        grid on;
        
        subplot(1,2,2);
        plot(t_exec');
        title('Execution Time (s)');
        grid on;
end